jQuery(document).ready(function($) {

    /**
     * 1. Helper: Toggle Full Page Loader
     * Blocks all user interaction by overlaying a spinner on the whole screen.
     */
    function toggleLoader(show = true) {
        if ($('.calculator-loader-wrapper').length === 0) {
            $('body').append('<div class="calculator-loader-wrapper"><div class="cs-loader"></div></div>');
        }

        if (show) {
            $('.calculator-loader-wrapper').css('display', 'flex');
            $('body').css('overflow', 'hidden'); // Prevent scrolling while loading
        } else {
            $('.calculator-loader-wrapper').hide();
            $('body').css('overflow', 'auto'); // Restore scrolling
        }
    }

    /**
     * 2. Helper: Update UI Panels
     * Injects the HTML returned from PHP into the appropriate containers.
     */
    function updateCalculatorUI(data) {
        // Ensure data exists
        if (!data) return;

        // 1. Update the sidebar
        if ($('.calculator-summary-content').length) {
            $('.calculator-summary-content').replaceWith(data.cartHtml);
        } else {
            // If sidebar container doesn't exist, you might need a target
            $('.calculator-container').prepend(data.cartHtml);
        }
        
        // 2. Update the item list (Middle Panel)
        $('.calculator-selected-item-list').html(data.itemsHtml);
        
        // 3. Trigger Menu Count Refresh
        $(document.body).trigger('wc_fragment_refresh');

        // 4. Show/Hide Detail Headers
        if (data.itemsHtml && data.itemsHtml.trim() !== '') {
            $('.calculator-item-details').show();
        } else {
            $('.calculator-item-details').hide();
        }

        toggleLoader(false);
    }

    /**
     * 3. Helper: Sync State to Server
     * This is the single "source of truth" for updating the WooCommerce cart.
     */
    function syncProductServices(parent_id, qty, servicesArray) {
        if (typeof ajax === 'undefined' || !ajax.url) {
            console.error("AJAX URL not found. Ensure wp_localize_script is configured.");
            return;
        }

        toggleLoader(true);
        
        $.post(ajax.url, {
            action: 'syncProductServices',
            product_parent_id: parent_id,
            quantity: qty,
            services: servicesArray
        }, function(res) {
            if (res.success) {
                updateCalculatorUI(res.data);
            } else {
                toggleLoader(false);
                console.error("Cart update failed.");
            }
        }).fail(function() {
            toggleLoader(false);
            alert("Error connecting to server. Please try again.");
        });
    }

    /**
     * 4. Helper: Gather Data and Trigger Sync
     * Figures out if a row is Simple or Variable and grabs the right values.
     */
    function handleAutoSync($element) {
        let li = $element.closest('li');
        let parent_id = li.data('parent');
        let qty = li.find('input[name="quantity"]').val();
        
        let services = [];
        let checkboxes = li.find('.service-check');

        if (checkboxes.length > 0) {
            // Variable product logic: Get checked checkboxes
            checkboxes.filter(':checked').each(function() {
                services.push($(this).val());
            });

            // If it's variable and user unchecked all services, remove from grid too
            if (services.length === 0) {
                resetGridSelection(parent_id);
            }
        } else {
            // Simple product logic: Use a flag so PHP knows it's not an empty variable update
            services = ['simple_product_flag']; 
        }

        syncProductServices(parent_id, qty, services);
    }

    /**
     * 5. Helper: Reset Grid UI Selection
     */
    function resetGridSelection(parent_id) {
        let gridItem = $('.product-item[data-parent="' + parent_id + '"]');
        gridItem.removeClass('selected-item');
        gridItem.find('.product-box').removeClass('checked');
        gridItem.find('.fa-check').removeClass('fa-check'); // Optional: remove check icon
    }

    // --- INITIALIZATION ---
    // On page load, fetch the current cart state and populate the UI
    if (typeof ajax !== 'undefined' && ajax.url) {
        $.post(ajax.url, { action: 'refresh_calculator' }, function(res) {
            if (res.success) updateCalculatorUI(res.data);
        });
    }

    // --- EVENT LISTENERS ---

    // 1. Grid Product Click (Add Initial Item)
    $('body').on('click', '.product-item', function() {
        let parent_id = $(this).data('parent');
        
        // Prevent double-clicking an already selected item
        if ($(this).hasClass('selected-item')) return;

        $(this).addClass('selected-item');
        $(this).find('.product-box').addClass('checked').find('i').addClass('fa-check');
        
        // Sync with initial Qty of 1 and a default service
        syncProductServices(parent_id, 1, ['washing']);
    });

    // 2. Service Checkbox Change (Auto-Sync)
    $('body').on('change', '.service-check', function() {
        handleAutoSync($(this));
    });

    // 3. Quantity Plus/Minus Buttons (Auto-Sync)
    $('body').on('click', '.qty-up, .qty-down', function() {
        let input = $(this).siblings('input');
        let currentVal = parseInt(input.val()) || 1;
        let newVal = $(this).hasClass('qty-up') ? currentVal + 1 : (currentVal > 1 ? currentVal - 1 : 1);
        
        input.val(newVal);
        handleAutoSync($(this));
    });

    // 4. Trash Icon (Full Removal)
    $('body').on('click', '.remove-all-services', function() {
        let li = $(this).closest('li');
        let parent_id = li.data('parent');
        
        resetGridSelection(parent_id);
        
        // Sending Qty 0 removes the item from the cart entirely
        syncProductServices(parent_id, 0, []);
    });

});