<?php
/**
 * Functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package WordPress
 * @subpackage Twenty_Twenty_One
 * @since Twenty Twenty-One 1.0
 */

// This theme requires WordPress 5.3 or later.
if ( version_compare( $GLOBALS['wp_version'], '5.3', '<' ) ) {
	require get_template_directory() . '/inc/back-compat.php';
}
add_theme_support('woocommerce');
if ( ! function_exists( 'twenty_twenty_one_setup' ) ) {
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 *
	 * @since Twenty Twenty-One 1.0
	 *
	 * @return void
	 */
	function twenty_twenty_one_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Twenty Twenty-One, use a find and replace
		 * to change 'twentytwentyone' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'twentytwentyone', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * This theme does not use a hard-coded <title> tag in the document head,
		 * WordPress will provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/**
		 * Add post-formats support.
		 */
		add_theme_support(
			'post-formats',
			array(
				'link',
				'aside',
				'gallery',
				'image',
				'quote',
				'status',
				'video',
				'audio',
				'chat',
			)
		);

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 1568, 9999 );

		register_nav_menus(
			array(
				'primary' => esc_html__( 'Primary menu', 'twentytwentyone' ),
				'footer'  => esc_html__( 'Secondary menu', 'twentytwentyone' ),
			)
		);

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support(
			'html5',
			array(
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
				'navigation-widgets',
			)
		);

		/*
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		$logo_width  = 300;
		$logo_height = 100;

		add_theme_support(
			'custom-logo',
			array(
				'height'               => $logo_height,
				'width'                => $logo_width,
				'flex-width'           => true,
				'flex-height'          => true,
				'unlink-homepage-logo' => true,
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );

		// Add support for editor styles.
		add_theme_support( 'editor-styles' );
		$background_color = get_theme_mod( 'background_color', 'D1E4DD' );
		if ( 127 > Twenty_Twenty_One_Custom_Colors::get_relative_luminance_from_hex( $background_color ) ) {
			add_theme_support( 'dark-editor-style' );
		}

		$editor_stylesheet_path = './assets/css/style-editor.css';

		// Note, the is_IE global variable is defined by WordPress and is used
		// to detect if the current browser is internet explorer.
		global $is_IE;
		if ( $is_IE ) {
			$editor_stylesheet_path = './assets/css/ie-editor.css';
		}

		// Enqueue editor styles.
		add_editor_style( $editor_stylesheet_path );

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__( 'Extra small', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'XS', 'Font size', 'twentytwentyone' ),
					'size'      => 16,
					'slug'      => 'extra-small',
				),
				array(
					'name'      => esc_html__( 'Small', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'S', 'Font size', 'twentytwentyone' ),
					'size'      => 18,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__( 'Normal', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'M', 'Font size', 'twentytwentyone' ),
					'size'      => 20,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__( 'Large', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'L', 'Font size', 'twentytwentyone' ),
					'size'      => 24,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__( 'Extra large', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'XL', 'Font size', 'twentytwentyone' ),
					'size'      => 40,
					'slug'      => 'extra-large',
				),
				array(
					'name'      => esc_html__( 'Huge', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'XXL', 'Font size', 'twentytwentyone' ),
					'size'      => 96,
					'slug'      => 'huge',
				),
				array(
					'name'      => esc_html__( 'Gigantic', 'twentytwentyone' ),
					'shortName' => esc_html_x( 'XXXL', 'Font size', 'twentytwentyone' ),
					'size'      => 144,
					'slug'      => 'gigantic',
				),
			)
		);

		// Custom background color.
		add_theme_support(
			'custom-background',
			array(
				'default-color' => 'd1e4dd',
			)
		);

		// Editor color palette.
		$black     = '#000000';
		$dark_gray = '#28303D';
		$gray      = '#39414D';
		$green     = '#D1E4DD';
		$blue      = '#D1DFE4';
		$purple    = '#D1D1E4';
		$red       = '#E4D1D1';
		$orange    = '#E4DAD1';
		$yellow    = '#EEEADD';
		$white     = '#FFFFFF';

		add_theme_support(
			'editor-color-palette',
			array(
				array(
					'name'  => esc_html__( 'Black', 'twentytwentyone' ),
					'slug'  => 'black',
					'color' => $black,
				),
				array(
					'name'  => esc_html__( 'Dark gray', 'twentytwentyone' ),
					'slug'  => 'dark-gray',
					'color' => $dark_gray,
				),
				array(
					'name'  => esc_html__( 'Gray', 'twentytwentyone' ),
					'slug'  => 'gray',
					'color' => $gray,
				),
				array(
					'name'  => esc_html__( 'Green', 'twentytwentyone' ),
					'slug'  => 'green',
					'color' => $green,
				),
				array(
					'name'  => esc_html__( 'Blue', 'twentytwentyone' ),
					'slug'  => 'blue',
					'color' => $blue,
				),
				array(
					'name'  => esc_html__( 'Purple', 'twentytwentyone' ),
					'slug'  => 'purple',
					'color' => $purple,
				),
				array(
					'name'  => esc_html__( 'Red', 'twentytwentyone' ),
					'slug'  => 'red',
					'color' => $red,
				),
				array(
					'name'  => esc_html__( 'Orange', 'twentytwentyone' ),
					'slug'  => 'orange',
					'color' => $orange,
				),
				array(
					'name'  => esc_html__( 'Yellow', 'twentytwentyone' ),
					'slug'  => 'yellow',
					'color' => $yellow,
				),
				array(
					'name'  => esc_html__( 'White', 'twentytwentyone' ),
					'slug'  => 'white',
					'color' => $white,
				),
			)
		);

		add_theme_support(
			'editor-gradient-presets',
			array(
				array(
					'name'     => esc_html__( 'Purple to yellow', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $purple . ' 0%, ' . $yellow . ' 100%)',
					'slug'     => 'purple-to-yellow',
				),
				array(
					'name'     => esc_html__( 'Yellow to purple', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $yellow . ' 0%, ' . $purple . ' 100%)',
					'slug'     => 'yellow-to-purple',
				),
				array(
					'name'     => esc_html__( 'Green to yellow', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $green . ' 0%, ' . $yellow . ' 100%)',
					'slug'     => 'green-to-yellow',
				),
				array(
					'name'     => esc_html__( 'Yellow to green', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $yellow . ' 0%, ' . $green . ' 100%)',
					'slug'     => 'yellow-to-green',
				),
				array(
					'name'     => esc_html__( 'Red to yellow', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $red . ' 0%, ' . $yellow . ' 100%)',
					'slug'     => 'red-to-yellow',
				),
				array(
					'name'     => esc_html__( 'Yellow to red', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $yellow . ' 0%, ' . $red . ' 100%)',
					'slug'     => 'yellow-to-red',
				),
				array(
					'name'     => esc_html__( 'Purple to red', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $purple . ' 0%, ' . $red . ' 100%)',
					'slug'     => 'purple-to-red',
				),
				array(
					'name'     => esc_html__( 'Red to purple', 'twentytwentyone' ),
					'gradient' => 'linear-gradient(160deg, ' . $red . ' 0%, ' . $purple . ' 100%)',
					'slug'     => 'red-to-purple',
				),
			)
		);

		/*
		* Adds starter content to highlight the theme on fresh sites.
		* This is done conditionally to avoid loading the starter content on every
		* page load, as it is a one-off operation only needed once in the customizer.
		*/
		if ( is_customize_preview() ) {
			require get_template_directory() . '/inc/starter-content.php';
			add_theme_support( 'starter-content', twenty_twenty_one_get_starter_content() );
		}

		// Add support for responsive embedded content.
		add_theme_support( 'responsive-embeds' );

		// Add support for custom line height controls.
		add_theme_support( 'custom-line-height' );

		// Add support for experimental link color control.
		add_theme_support( 'experimental-link-color' );

		// Add support for experimental cover block spacing.
		add_theme_support( 'custom-spacing' );

		// Add support for custom units.
		// This was removed in WordPress 5.6 but is still required to properly support WP 5.5.
		add_theme_support( 'custom-units' );

		// Remove feed icon link from legacy RSS widget.
		add_filter( 'rss_widget_feed_link', '__return_false' );
	}
}
add_action( 'after_setup_theme', 'twenty_twenty_one_setup' );

/**
 * Register widget area.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 *
 * @return void
 */
function twenty_twenty_one_widgets_init() {

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer', 'twentytwentyone' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'twentytwentyone' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
	register_sidebar(
		array(
			'name'          => esc_html__( 'Contact', 'twentytwentyone' ),
			'id'            => 'contact',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'twentytwentyone' ),
			'before_widget' => '',
			'after_widget'  => '',
			'before_title'  => '',
			'after_title'   => '',
		)
	);
}
add_action( 'widgets_init', 'twenty_twenty_one_widgets_init' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @global int $content_width Content width.
 *
 * @return void
 */
function twenty_twenty_one_content_width() {
	// This variable is intended to be overruled from themes.
	// Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'twenty_twenty_one_content_width', 750 );
}
add_action( 'after_setup_theme', 'twenty_twenty_one_content_width', 0 );

/**
 * Enqueue scripts and styles.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twenty_twenty_one_scripts() {
	// Note, the is_IE global variable is defined by WordPress and is used
	// to detect if the current browser is internet explorer.
	global $is_IE, $wp_scripts;
	if ( $is_IE ) {
		// If IE 11 or below, use a flattened stylesheet with static values replacing CSS Variables.
		wp_enqueue_style( 'twenty-twenty-one-style', get_template_directory_uri() . '/assets/css/ie.css', array(), wp_get_theme()->get( 'Version' ) );
	} else {
		// If not IE, use the standard stylesheet.
		//wp_enqueue_style( 'twenty-twenty-one-style', get_template_directory_uri() . '/style.css', array(), wp_get_theme()->get( 'Version' ) );
	}
		wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.css');
		wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/css/font-awesome.min.css');
		wp_enqueue_style( 'slick', get_template_directory_uri() . '/assets/css/slick.css');
		wp_enqueue_style( 'slick-theme', get_template_directory_uri() . '/assets/css/slick-theme.css');
		wp_enqueue_style( 'screen', get_template_directory_uri() . '/assets/css/style.css');

		wp_enqueue_script('jquery',get_template_directory_uri().'/assets/js/jquery.min.js',array(),'',false);
		wp_enqueue_script('bootstrap-js',get_template_directory_uri().'/assets/js/bootstrap.js',array(),'',true);
		wp_enqueue_script('slick-js',get_template_directory_uri().'/assets/js/slick.js',array(),'',true);
		wp_enqueue_script('custom',get_template_directory_uri().'/assets/js/custom.js',array(),'',true);
		






	// RTL styles.
	wp_style_add_data( 'twenty-twenty-one-style', 'rtl', 'replace' );

	// Print styles.
	wp_enqueue_style( 'twenty-twenty-one-print-style', get_template_directory_uri() . '/assets/css/print.css', array(), wp_get_theme()->get( 'Version' ), 'print' );

	// Threaded comment reply styles.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	// Register the IE11 polyfill file.
	wp_register_script(
		'twenty-twenty-one-ie11-polyfills-asset',
		get_template_directory_uri() . '/assets/js/polyfills.js',
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);

	// Register the IE11 polyfill loader.
	wp_register_script(
		'twenty-twenty-one-ie11-polyfills',
		null,
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);
	wp_add_inline_script(
		'twenty-twenty-one-ie11-polyfills',
		wp_get_script_polyfill(
			$wp_scripts,
			array(
				'Element.prototype.matches && Element.prototype.closest && window.NodeList && NodeList.prototype.forEach' => 'twenty-twenty-one-ie11-polyfills-asset',
			)
		)
	);

	// Main navigation scripts.
	if ( has_nav_menu( 'primary' ) ) {
		wp_enqueue_script(
			'twenty-twenty-one-primary-navigation-script',
			get_template_directory_uri() . '/assets/js/primary-navigation.js',
			array( 'twenty-twenty-one-ie11-polyfills' ),
			wp_get_theme()->get( 'Version' ),
			true
		);
	}

	// Responsive embeds script.
	wp_enqueue_script(
		'twenty-twenty-one-responsive-embeds-script',
		get_template_directory_uri() . '/assets/js/responsive-embeds.js',
		array( 'twenty-twenty-one-ie11-polyfills' ),
		wp_get_theme()->get( 'Version' ),
		true
	);
}
add_action( 'wp_enqueue_scripts', 'twenty_twenty_one_scripts' );

/**
 * Enqueue block editor script.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twentytwentyone_block_editor_script() {

	wp_enqueue_script( 'twentytwentyone-editor', get_theme_file_uri( '/assets/js/editor.js' ), array( 'wp-blocks', 'wp-dom' ), wp_get_theme()->get( 'Version' ), true );
}

add_action( 'enqueue_block_editor_assets', 'twentytwentyone_block_editor_script' );

/**
 * Fix skip link focus in IE11.
 *
 * This does not enqueue the script because it is tiny and because it is only for IE11,
 * thus it does not warrant having an entire dedicated blocking script being loaded.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @link https://git.io/vWdr2
 */
function twenty_twenty_one_skip_link_focus_fix() {

	// If SCRIPT_DEBUG is defined and true, print the unminified file.
	if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) {
		echo '<script>';
		include get_template_directory() . '/assets/js/skip-link-focus-fix.js';
		echo '</script>';
	} else {
		// The following is minified via `npx terser --compress --mangle -- assets/js/skip-link-focus-fix.js`.
		?>
		<script>
		/(trident|msie)/i.test(navigator.userAgent)&&document.getElementById&&window.addEventListener&&window.addEventListener("hashchange",(function(){var t,e=location.hash.substring(1);/^[A-z0-9_-]+$/.test(e)&&(t=document.getElementById(e))&&(/^(?:a|select|input|button|textarea)$/i.test(t.tagName)||(t.tabIndex=-1),t.focus())}),!1);
		</script>
		<?php
	}
}
add_action( 'wp_print_footer_scripts', 'twenty_twenty_one_skip_link_focus_fix' );

/**
 * Enqueue non-latin language styles.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twenty_twenty_one_non_latin_languages() {
	$custom_css = twenty_twenty_one_get_non_latin_css( 'front-end' );

	if ( $custom_css ) {
		wp_add_inline_style( 'twenty-twenty-one-style', $custom_css );
	}
}
add_action( 'wp_enqueue_scripts', 'twenty_twenty_one_non_latin_languages' );

// SVG Icons class.
require get_template_directory() . '/classes/class-twenty-twenty-one-svg-icons.php';

// Custom color classes.
require get_template_directory() . '/classes/class-twenty-twenty-one-custom-colors.php';
new Twenty_Twenty_One_Custom_Colors();

// Enhance the theme by hooking into WordPress.
require get_template_directory() . '/inc/template-functions.php';

// Menu functions and filters.
require get_template_directory() . '/inc/menu-functions.php';

// Custom template tags for the theme.
require get_template_directory() . '/inc/template-tags.php';

// Customizer additions.
require get_template_directory() . '/classes/class-twenty-twenty-one-customize.php';
new Twenty_Twenty_One_Customize();

// Block Patterns.
require get_template_directory() . '/inc/block-patterns.php';

// Block Styles.
require get_template_directory() . '/inc/block-styles.php';

// Dark Mode.
require_once get_template_directory() . '/classes/class-twenty-twenty-one-dark-mode.php';
new Twenty_Twenty_One_Dark_Mode();

/**
 * Enqueue scripts for the customizer preview.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twentytwentyone_customize_preview_init() {
	wp_enqueue_script(
		'twentytwentyone-customize-helpers',
		get_theme_file_uri( '/assets/js/customize-helpers.js' ),
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);

	wp_enqueue_script(
		'twentytwentyone-customize-preview',
		get_theme_file_uri( '/assets/js/customize-preview.js' ),
		array( 'customize-preview', 'customize-selective-refresh', 'jquery', 'twentytwentyone-customize-helpers' ),
		wp_get_theme()->get( 'Version' ),
		true
	);
}
add_action( 'customize_preview_init', 'twentytwentyone_customize_preview_init' );

/**
 * Enqueue scripts for the customizer.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twentytwentyone_customize_controls_enqueue_scripts() {

	wp_enqueue_script(
		'twentytwentyone-customize-helpers',
		get_theme_file_uri( '/assets/js/customize-helpers.js' ),
		array(),
		wp_get_theme()->get( 'Version' ),
		true
	);
}
add_action( 'customize_controls_enqueue_scripts', 'twentytwentyone_customize_controls_enqueue_scripts' );

/**
 * Calculate classes for the main <html> element.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twentytwentyone_the_html_classes() {
	/**
	 * Filters the classes for the main <html> element.
	 *
	 * @since Twenty Twenty-One 1.0
	 *
	 * @param string The list of classes. Default empty string.
	 */
	$classes = apply_filters( 'twentytwentyone_html_classes', '' );
	if ( ! $classes ) {
		return;
	}
	echo 'class="' . esc_attr( $classes ) . '"';
}

/**
 * Add "is-IE" class to body if the user is on Internet Explorer.
 *
 * @since Twenty Twenty-One 1.0
 *
 * @return void
 */
function twentytwentyone_add_ie_class() {
	?>
	<script>
	if ( -1 !== navigator.userAgent.indexOf( 'MSIE' ) || -1 !== navigator.appVersion.indexOf( 'Trident/' ) ) {
		document.body.classList.add( 'is-IE' );
	}
	</script>
	<?php
}
add_action( 'wp_footer', 'twentytwentyone_add_ie_class' );

if ( ! function_exists( 'wp_get_list_item_separator' ) ) :
	/**
	 * Retrieves the list item separator based on the locale.
	 *
	 * Added for backward compatibility to support pre-6.0.0 WordPress versions.
	 *
	 * @since 6.0.0
	 */
	function wp_get_list_item_separator() {
		/* translators: Used between list items, there is a space after the comma. */
		return __( ', ', 'twentytwentyone' );
	}
endif;
include('wp_bootstrap_navwalker.php');

/* Disable Gutenberg Block Editor */
add_filter('use_block_editor_for_post', '__return_false', 10);

/* Disable Widgets Block Editor */
add_filter( 'use_widgets_block_editor', '__return_false' );

remove_action ('woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart', 30);

add_action ('woocommerce_single_product_summary', 'woocommerce_template_loop_add_to_cart', 30);

add_filter( 'woocommerce_add_to_cart_fragments', 'wc_refresh_cart_fragments', 50, 1 );
function wc_refresh_cart_fragments( $fragments ){
    $cart = WC()->cart;
    $count = 0;

    if ( $cart ) {
        $product_ids = [];
        foreach ( $cart->get_cart() as $cart_item ) {
            // Collect the parent product ID
            $product_ids[] = $cart_item['product_id'];
        }
        // Remove duplicates and count
        $count = count( array_unique( $product_ids ) );
    }

    // Normal version - Force the count to be our new calculated $count
    $count_normal = '<a class="cart-content">' . $count . '</a>';
    $fragments['.cart-content'] = $count_normal;

    return $fragments;
}

/**
 * Core filter to force WooCommerce to report unique product count
 */
add_filter( 'woocommerce_cart_contents_count', 'force_unique_cart_count' );
function force_unique_cart_count( $count ) {
    if ( ! WC()->cart ) return $count;

    $product_ids = [];
    foreach ( WC()->cart->get_cart() as $cart_item ) {
        $product_ids[] = $cart_item['product_id'];
    }
    return count( array_unique( $product_ids ) );
}

function local() {

    wp_localize_script( 'js-file-handle', 'ajax', array(
        'url' => admin_url( 'admin-ajax.php' )
    ) );

}

add_action('admin_enqueue_scripts', 'local');
add_action('wp_enqueue_scripts', 'local');

add_filter( 'woocommerce_account_menu_items', 'misha_remove_my_account_links' );
function misha_remove_my_account_links( $menu_links ){

    //unset( $menu_links[ 'edit-address' ] ); // Addresses

    //unset( $menu_links[ 'dashboard' ] ); // Remove Dashboard
    //unset( $menu_links[ 'payment-methods' ] ); // Remove Payment Methods
    //unset( $menu_links[ 'orders' ] ); // Remove Orders
    unset( $menu_links[ 'downloads' ] ); // Disable Downloads
    //unset( $menu_links[ 'edit-account' ] ); // Remove Account details tab
    //unset( $menu_links[ 'customer-logout' ] ); // Remove Logout link

    return $menu_links;

}

// Remove some checkout billing fields
function kia_filter_billing_fields($fields){
    unset( $fields["billing_country"] );
    unset( $fields["billing_company"] );
    //unset( $fields["billing_address_1"] );
    unset( $fields["billing_address_2"] );
    //unset( $fields["billing_city"] );
    unset( $fields["billing_state"] );
    unset( $fields["billing_postcode"] );
    //unset( $fields["billing_phone"] );
    return $fields;
}
add_filter( 'woocommerce_billing_fields', 'kia_filter_billing_fields' );



// Enqueue JS for Calculator
function enqueue_calculator_scripts() {
    wp_enqueue_script('calculator', get_template_directory_uri() . '/assets/js/calculator.js', array('jquery'), '1.0', true);

    wp_localize_script('calculator', 'ajax', array(
        'url' => admin_url('admin-ajax.php')
    ));

    wp_enqueue_style('multiselect-css', 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-multiselect/0.9.13/css/bootstrap-multiselect.css');
    wp_enqueue_script('multiselect-js', 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-multiselect/0.9.13/js/bootstrap-multiselect.js', array('jquery'), '0.9.13', true);
}
add_action('wp_enqueue_scripts', 'enqueue_calculator_scripts');


add_filter( 'woocommerce_product_get_price', 'sync_defence_pricing', 99, 2 );
add_filter( 'woocommerce_product_variation_get_price', 'sync_defence_pricing', 99, 2 );

function sync_defence_pricing( $price, $product ) {
    // If the price is empty or not a valid number, return it as is
    if ( '' === $price || ! is_numeric( $price ) ) {
        return $price;
    }

    // Check if we are in the 'Defence' session
    if ( isset( $_SESSION['area'] ) && $_SESSION['area'] === 'defence' ) {
        $multiplier = 1.2; // Your 20% increase
        
        // Convert to float to be safe and multiply
        return (float) $price * $multiplier;
    }

    return $price;
}


/**
 * 1. AJAX HANDLER: Sync Product Services
 * Handles adding/removing/updating both Variable and Simple products.
 */
add_action('wp_ajax_syncProductServices', 'syncProductServices');
add_action('wp_ajax_nopriv_syncProductServices', 'syncProductServices');
/**
 * 1. AJAX HANDLER: Sync Product Services
 */
function syncProductServices() {
    $parent_id = absint($_POST['product_parent_id']);
    $qty = max(0, absint($_POST['quantity']));
    $selected_services = $_POST['services'] ?? [];

    if ( WC()->cart ) {
        // Remove existing items for this parent
        foreach (WC()->cart->get_cart() as $key => $item) {
            if ($item['product_id'] == $parent_id) {
                WC()->cart->remove_cart_item($key);
            }
        }

        // Re-add items if quantity > 0
        if ($qty > 0) {
            $product = wc_get_product($parent_id);

            if ($product && $product->is_type('simple')) {
                WC()->cart->add_to_cart($parent_id, $qty);
            } elseif ($product && $product->is_type('variable')) {
                foreach ($selected_services as $slug) {
                    if ($slug === 'simple_product_flag') continue;
                    
                    $vars = ['attribute_pa_classification' => $slug];
                    $data_store = WC_Data_Store::load('product');
                    $variation_id = $data_store->find_matching_product_variation($product, $vars);
                    
                    if ($variation_id) {
                        WC()->cart->add_to_cart($parent_id, $qty, $variation_id, $vars);
                    }
                }
            }
        }
        // Force refresh cart data
        WC()->cart->calculate_totals();
        
        // 2. Refresh the session so the persistent price filter (above) locks in
        WC()->session->set('cart', WC()->cart->get_cart_for_session());
    }

    // SUCCESS: Return both HTML panels
    wp_send_json_success([
        'cartHtml'  => generate_calculator_cart_html(),
        'itemsHtml' => generate_selected_items_html()
    ]);
}

/**
 * 2. AJAX HANDLER: Refresh Calculator
 */
add_action('wp_ajax_refresh_calculator', 'refresh_calculator');
add_action('wp_ajax_nopriv_refresh_calculator', 'refresh_calculator');
function refresh_calculator() {
    if ( WC()->cart ) {
        WC()->cart->calculate_totals();
    }

    wp_send_json_success([
        'cartHtml'  => generate_calculator_cart_html(),
        'itemsHtml' => generate_selected_items_html()
    ]);
}

/**
 * 3. UI GENERATOR: Order Summary Panel
 */
function generate_calculator_cart_html() {
    $cart = WC()->cart;
    if (!$cart || $cart->is_empty()) {
        return '<div class="calculator-summary-content"><h3>Order Summary</h3><p>No items selected.</p></div>';
    }

    $html = '<div class="calculator-summary-content">';
    $html .= '  <h3 class="summary-title" style="margin-top:0; border-bottom:2px solid #79b927; padding-bottom:10px;">Order Summary</h3>';
    $html .= '  <ul class="summary-items-list" style="list-style:none; padding:0; margin-bottom:15px;">';
    
    $grouped_display = [];
    $service_subtotal = 0; // Track services only
    
    // Check for Area once
    $area = isset($_COOKIE['selected_laundry_area']) ? $_COOKIE['selected_laundry_area'] : 'Other Area';
    $is_defence = ($area === 'Defence');
    $delivery_fee = $is_defence ? 200 : 100;

    foreach ($cart->get_cart() as $cart_item) {
        $p_id = $cart_item['product_id'];
        
        if (!isset($grouped_display[$p_id])) {
            $grouped_display[$p_id] = [
                'name' => get_the_title($p_id),
                'qty'  => $cart_item['quantity'],
                'services' => [],
                'individual_prices' => [],
                'item_combined_total' => 0
            ];
        }
        
        // Pull regular price to avoid snowballing
        $base_price = $cart_item['data']->get_regular_price();
        $final_unit_price = $is_defence ? ($base_price * 1.5) : $base_price;
        
        $line_total = ($final_unit_price * $cart_item['quantity']);
        $grouped_display[$p_id]['item_combined_total'] += $line_total;
        $grouped_display[$p_id]['individual_prices'][] = wc_price($final_unit_price);
        
        $service_subtotal += $line_total;

        if (!empty($cart_item['variation']) && isset($cart_item['variation']['attribute_pa_classification'])) {
            $term = get_term_by('slug', $cart_item['variation']['attribute_pa_classification'], 'pa_classification');
            if ($term) $grouped_display[$p_id]['services'][] = $term->name;
        }
    }

    foreach ($grouped_display as $item) {
        $html .= ' <li style="margin-bottom:15px; border-bottom:1px solid #eee; padding-bottom:10px;">';
        $html .= '    <div style="display:flex; justify-content:space-between; font-weight:bold;">';
        $html .= '       <span>' . $item['name'] . '</span> <span>x' . $item['qty'] . '</span>';
        $html .= '    </div>';
        
        if (!empty($item['services'])) {
            $html .= '    <div style="font-size:13px; color:#79b927; font-weight:600; margin-top:4px;">' . implode(' + ', array_unique($item['services'])) . '</div>';
            $html .= '    <div style="font-size:12px; color:#999; font-style:italic;">(' . implode(' + ', $item['individual_prices']) . ')</div>';
        }

        $html .= '    <div style="text-align:right; font-weight:600; font-size:14px; margin-top:5px;">Subtotal: ' . wc_price($item['item_combined_total']) . '</div>';
        $html .= ' </li>';
    }
    
    $html .= '  </ul>';

    // Calculation for Grand Total
    $total_estimate = $service_subtotal + $delivery_fee;

    $html .= '  <div class="summary-total-section" style="background:#f9f9f9; padding:15px; border-radius:5px;">';
    
    // Line 1: Service Subtotal
    $html .= '      <div style="display:flex; justify-content:space-between; font-size:14px; margin-bottom:5px; color:#333;">';
    $html .= '          <span>Service Total:</span> <span>' . wc_price($service_subtotal) . '</span>';
    $html .= '      </div>';

    // Line 2: Delivery Fee
    $html .= '      <div style="display:flex; justify-content:space-between; font-size:14px; margin-bottom:10px; color:#666; font-style:italic;">';
    $html .= '          <span>Delivery (' . $area . '):</span> <span>' . wc_price($delivery_fee) . '</span>';
    $html .= '      </div>';

    // Line 3: Final Total
    $html .= '      <div style="display:flex; justify-content:space-between; font-weight:bold; font-size:1.3em; color:#79b927; border-top:1px solid #ddd; padding-top:10px;">';
    $html .= '          <span>Total Estimate:</span> <span>' . wc_price($total_estimate) . '</span>';
    $html .= '      </div>';
    
    if ($is_defence) {
        $html .= '<div style="font-size:11px; color:#888; margin-top:5px; text-align:center;">(Includes 50% Defence Area Premium)</div>';
    }

    $html .= '      <a href="'.wc_get_checkout_url().'" class="btn btn-success" style="display:block; margin-top:15px; width:100%; background:#79b927; border:none; color:#fff; text-align:center; padding:10px; border-radius:4px; font-weight:bold; text-decoration:none;">Book Now</a>';
    $html .= '  </div>';
    $html .= '</div>';

    return $html;
}

/**
 * 4. UI GENERATOR: Selected Items List (Middle Panel)
 */
function generate_selected_items_html() {
    $html = '';
    $cart = WC()->cart;
    if (!$cart || $cart->is_empty()) return $html;

    // Group cart items by their Parent Product ID
    $grouped = [];
    foreach ($cart->get_cart() as $item) {
        $p_id = $item['product_id'];
        if (!isset($grouped[$p_id])) {
            $grouped[$p_id] = [
                'qty' => $item['quantity'],
                'active_vars' => []
            ];
        }
        // Collect the variation slugs (e.g., 'washing', 'ironing')
        if (!empty($item['variation']) && isset($item['variation']['attribute_pa_classification'])) {
            $grouped[$p_id]['active_vars'][] = $item['variation']['attribute_pa_classification'];
        }
    }

    foreach ($grouped as $parent_id => $data) {
        $product = wc_get_product($parent_id);
        if (!$product) continue;
        
        $html .= '<li class="calculator-row" data-parent="'.$parent_id.'" style="display:flex; align-items:center; gap:15px; padding:15px; border-bottom:1px solid #f0f0f0;">
            <div style="flex:1;"><strong>'.get_the_title($parent_id).'</strong></div>
            
            <div class="csi-quantity" style="flex:1; display:flex; align-items:center; gap:10px;">
                <span class="qty-down" style="cursor:pointer; color:#888;"><i class="fa fa-minus-circle"></i></span>
                <input type="number" name="quantity" value="'.$data['qty'].'" readonly style="width:45px; text-align:center; border:1px solid #ddd; border-radius:4px;">
                <span class="qty-up" style="cursor:pointer; color:#28a745;"><i class="fa fa-plus-circle"></i></span>
            </div>';

        // Checkbox area for Variable Products
        if ($product->is_type('variable')) {
            $html .= '<div class="csi-service-checkboxes" style="flex:2; display:flex; gap:10px;">';
            $terms = wc_get_product_terms($parent_id, 'pa_classification', ['fields'=>'all']);
            foreach ($terms as $term) {
                $checked = in_array($term->slug, $data['active_vars']) ? 'checked' : '';
                $html .= '<label style="font-size:12px; cursor:pointer; margin-bottom:0;">
                            <input type="checkbox" class="service-check" value="'.$term->slug.'" '.$checked.'> '.$term->name.'
                          </label>';
            }
            $html .= '</div>';
        } else {
            // Label for Simple Products
            $html .= '<div style="flex:2; font-size:12px; color:#999;">Standard Service</div>';
        }

        $html .= '<div class="csi-delete" style="flex:0.2; text-align:right;">
                <i class="fa fa-trash-o remove-all-services" style="cursor:pointer; color:#e74c3c; font-size:18px;"></i>
            </div>
        </li>';
    }
    return $html;
}

/**
 * 5. CART PAGE: Visually Group Variations
 */
add_filter( 'woocommerce_cart_item_name', 'group_calculator_items_on_cart_page', 10, 3 );
function group_calculator_items_on_cart_page( $name, $cart_item, $cart_item_key ) {
    if ( ! is_cart() && ! is_checkout() ) return $name;

    $product_id = $cart_item['product_id'];
    $cart = WC()->cart;
    if (!$cart) return $name;
    
    $all_cart_items = $cart->get_cart();
    $services = [];

    foreach ( $all_cart_items as $item ) {
        if ( $item['product_id'] == $product_id ) {
            if ( !empty($item['variation']) && isset($item['variation']['attribute_pa_classification']) ) {
                $term = get_term_by('slug', $item['variation']['attribute_pa_classification'], 'pa_classification');
                if ($term) $services[] = $term->name;
            }
        }
    }

    static $already_processed = [];
    if ( in_array($product_id, $already_processed) ) return '';
    
    $already_processed[] = $product_id;

    if ( !empty($services) ) {
        $name .= '<div class="cart-service-list" style="font-size: 0.85em; color: #28a745; font-weight: 600;">' . implode( ' + ', array_unique($services) ) . '</div>';
    }

    return $name;
}

add_filter( 'woocommerce_cart_item_visible', 'hide_duplicate_calculator_rows', 10, 3 );
function hide_duplicate_calculator_rows( $visible, $cart_item, $cart_item_key ) {
    static $processed_ids = [];
    $product_id = $cart_item['product_id'];
    if ( in_array( $product_id, $processed_ids ) ) return false;
    $processed_ids[] = $product_id;
    return $visible;
}



/**
 * 1. CART PAGE: Combined Title, Services, and Price Breakdown
 * This puts Title, Service list, and (Price + Price) in one column.
 */
add_filter( 'woocommerce_cart_item_name', 'cart_page_item_combined_display', 10, 3 );
function cart_page_item_combined_display( $name, $cart_item, $cart_item_key ) {
    if ( ! is_cart() && ! is_checkout() ) return $name;

    $product_id = $cart_item['product_id'];
    $cart = WC()->cart;
    $services = [];
    $prices = [];

    // Gather unique services and prices for this product group
    foreach ( $cart->get_cart() as $item ) {
        if ( $item['product_id'] == $product_id ) {
            if ( !empty($item['variation']) && isset($item['variation']['attribute_pa_classification']) ) {
                $term = get_term_by('slug', $item['variation']['attribute_pa_classification'], 'pa_classification');
                if ($term) $services[] = $term->name;
            }
            $prices[] = wc_price($item['data']->get_price());
        }
    }

    // Static tracker to prevent duplicate rows
    static $name_processed = [];
    if ( in_array($product_id, $name_processed) ) return '';
    $name_processed[] = $product_id;

    // Get clean product name (remove variation suffix)
    $product = wc_get_product($product_id);
    $clean_name = $product->get_name();

    $output = '<span class="cart-product-title" style="font-weight:bold; font-size:1.1em;">' . $clean_name . '</span>';

    if ( !empty($services) ) {
        // Unique services: Ironing + Washing
        $output .= '<div class="cart-service-list" style="font-size: 0.9em; color: #28a745; font-weight: 600; margin-top: 5px;">' . implode( ' + ', array_unique($services) ) . '</div>';
        
        // Price Breakdown: (₨10.00 + ₨20.00)
        $output .= '<div class="cart-price-breakdown" style="font-size: 0.85em; color: #888; font-style: italic;">(' . implode(' + ', $prices) . ')</div>';
    }

    return $output;
}

/**
 * 2. CART PAGE: Completely Remove Variation Metadata & "Classification" labels
 * This stops "Classification: Ironing" and the extra variation lines.
 */
add_filter( 'woocommerce_get_item_data', 'remove_variation_metadata_from_cart', 10, 2 );
function remove_variation_metadata_from_cart( $data, $cart_item ) {
    // This empties the array that holds the "Classification: Ironing" text
    return array();
}

// Remove variation suffix from the title (T-Shirts - Ironing -> T-Shirts)
add_filter( 'woocommerce_is_attribute_in_product_name', '__return_false' );

/**
 * 3. CART PAGE: Simple Unit Price Column
 */
add_filter( 'woocommerce_cart_item_price', 'cart_page_item_clean_price', 10, 3 );
function cart_page_item_clean_price( $price, $cart_item, $cart_item_key ) {
    $product_id = $cart_item['product_id'];
    $total_unit_price = 0;

    foreach ( WC()->cart->get_cart() as $item ) {
        if ( $item['product_id'] == $product_id ) {
            $total_unit_price += $item['data']->get_price();
        }
    }

    static $price_processed = [];
    if ( in_array($product_id, $price_processed) ) return '';
    $price_processed[] = $product_id;

    return '<strong>' . wc_price($total_unit_price) . '</strong>';
}

/**
 * 4. CART PAGE: Subtotal Column
 */
add_filter( 'woocommerce_cart_item_subtotal', 'cart_page_item_subtotal_grouped', 10, 3 );
function cart_page_item_subtotal_grouped( $subtotal, $cart_item, $cart_item_key ) {
    $product_id = $cart_item['product_id'];
    $row_total = 0;

    foreach ( WC()->cart->get_cart() as $item ) {
        if ( $item['product_id'] == $product_id ) {
            $row_total += ( $item['data']->get_price() * $item['quantity'] );
        }
    }

    static $subtotal_processed = [];
    if ( in_array($product_id, $subtotal_processed) ) return '';
    $subtotal_processed[] = $product_id;

    return '<strong style="color:#28a745;">' . wc_price($row_total) . '</strong>';
}

/**
 * 5. CART PAGE: Row Visibility Logic
 */
add_filter( 'woocommerce_cart_item_visible', 'hide_extra_cart_rows', 10, 3 );
function hide_extra_cart_rows( $visible, $cart_item, $cart_item_key ) {
    static $visible_processed = [];
    $product_id = $cart_item['product_id'];
    if ( in_array( $product_id, $visible_processed ) ) return false;
    $visible_processed[] = $product_id;
    return $visible;
}

/**
 * 1. CLEAN GROUPING: Merges names but lets the table handle Quantity/Total
 */
add_filter( 'woocommerce_cart_item_name', 'laundry_clean_checkout_grouping', 999, 3 );
function laundry_clean_checkout_grouping( $name, $cart_item, $cart_item_key ) {
    if ( ! is_cart() && ! is_checkout() ) return $name;

    $product_id = $cart_item['product_id'];
    static $processed_ids = [];

    // Return a marker to hide duplicate rows via CSS
    if ( in_array( $product_id, $processed_ids ) ) {
        return '<span class="hide-duplicate-row"></span>'; 
    }
    $processed_ids[] = $product_id;

    $services = [];
    $prices = [];
    
    // Gather services and unit prices only
    foreach ( WC()->cart->get_cart() as $item ) {
        if ( $item['product_id'] == $product_id ) {
            if ( ! empty( $item['variation'] ) && isset( $item['variation']['attribute_pa_classification'] ) ) {
                $term = get_term_by( 'slug', $item['variation']['attribute_pa_classification'], 'pa_classification' );
                if ( $term ) $services[] = $term->name;
            }
            $prices[] = wc_price( $item['data']->get_price() );
        }
    }

    $output = '<div class="laundry-item-meta">';
    $output .= '<strong style="font-size: 1.1em; display: block;">' . get_the_title( $product_id ) . '</strong>';
    
    if ( ! empty( $services ) ) {
        // Show: Ironing + Washing
        $output .= '<div style="color: #28a745; font-weight: 600; font-size: 0.9em; margin-top: 4px;">' 
                   . implode( ' + ', array_unique( $services ) ) . '</div>';
        // Show: (₨50.00 + ₨50.00)
        $output .= '<div style="color: #888; font-size: 0.85em; font-style: italic;">(' 
                   . implode( ' + ', $prices ) . ')</div>';
    }
    $output .= '</div>';

    return $output;
}

/**
 * 2. VISIBILITY: Tell WooCommerce which rows to actually draw
 */
add_filter( 'woocommerce_cart_item_visible', 'laundry_hide_extra_rows', 999, 3 );
function laundry_hide_extra_rows( $visible, $cart_item, $cart_item_key ) {
    static $visible_tracker = [];
    $product_id = $cart_item['product_id'];
    if ( in_array( $product_id, $visible_tracker ) ) return false;
    $visible_tracker[] = $product_id;
    return $visible;
}

// Clean up product name suffixes
add_filter( 'woocommerce_is_attribute_in_product_name', '__return_false' );


// Part 1: Group the items and sum the prices
add_filter( 'woocommerce_order_get_items', 'final_bulletproof_grouping', 10, 2 );
function final_bulletproof_grouping( $items, $order ) {
    if ( ! is_object( $order ) ) return $items;

    static $is_processing = false;
    if ( $is_processing ) return $items;
    $is_processing = true;

    $grouped = [];
    $other_items = [];

    foreach ( $items as $item_id => $item ) {
        if ( ! is_a( $item, 'WC_Order_Item_Product' ) ) {
            $other_items[$item_id] = $item;
            continue;
        }

        $p_id = $item->get_product_id();
        $raw_name = $item->get_name();
        
        // Identify the service
        $service = (stripos($raw_name, 'Folding') !== false) ? 'Folding' : 
                   ((stripos($raw_name, 'Ironing') !== false) ? 'Ironing' : 'Washing');

        if ( ! isset( $grouped[$p_id] ) ) {
            $product = $item->get_product();
            $item->set_name( $product ? $product->get_title() : $raw_name );
            $item->update_meta_data( '_temp_list', [$service] );
            $grouped[$p_id] = $item;
        } else {
            $main = $grouped[$p_id];
            
            // Sum prices correctly
            $new_total = $main->get_total() + $item->get_total();
            $main->set_total( $new_total );
            $main->set_subtotal( $new_total ); // Fixes the "discount" display
            
            // Merge services
            $services = (array) $main->get_meta( '_temp_list' );
            if ( !in_array( $service, $services ) ) {
                $services[] = $service;
                $main->update_meta_data( '_temp_list', $services );
            }
        }
    }

    // Finalize names and remove the "extra" meta lines
    foreach ( $grouped as $item ) {
        $services = $item->get_meta( '_temp_list' );
        if ( !empty( $services ) ) {
            $item->set_name( $item->get_name() . ' (' . implode( ', ', $services ) . ')' );
        }
        // Force-delete the variation meta so it doesn't show up below the name
        $item->delete_meta_data('pa_classification');
        $item->delete_meta_data('classification');
    }

    $is_processing = false;
    return $grouped + $other_items;
}

// Part 2: Force the services to appear under the title
add_filter( 'woocommerce_order_item_get_formatted_meta_data', 'force_show_laundry_services', 10, 2 );
function force_show_laundry_services( $formatted_meta, $item ) {
    if ( is_admin() ) return $formatted_meta;

    // Get that temporary list we made in Part 1
    $services = $item->get_meta( '_temp_services_list' );

    if ( !empty($services) ) {
        // Build a fresh meta object that WooCommerce is forced to display
        $formatted_meta[] = (object) array(
            'key'           => 'services_list',
            'display_key'   => 'Services', // This is the label the user sees
            'display_value' => implode( ', ', $services ), // E.g., Washing, Ironing
            'label'         => 'Services',
            'value'         => implode( ', ', $services ),
        );
    }

    return $formatted_meta;
}

add_filter( 'woocommerce_order_shipping_to_display', 'clean_shipping_display_name', 10, 2 );
function clean_shipping_display_name( $shipping, $order ) {
    // If the shipping name is too long or contains repetitions, just show "Standard Shipping" 
    // or the simple method name.
    return $order->get_shipping_method(); 
}

/**
 * Force grouping display in the WooCommerce Admin Order Edit Screen
 */
add_action( 'woocommerce_before_order_itemmeta', 'laundry_admin_force_group_display', 10, 3 );
function laundry_admin_force_group_display( $item_id, $item, $product ) {
    if ( ! is_admin() ) return;

    // Pull the services gathered in the grouping filter
    $services = $item->get_meta('_temp_list'); 
    
    if ( ! empty( $services ) && is_array( $services ) ) {
        echo '<div class="laundry-admin-services" style="color:#28a745; font-weight:bold; border-left: 3px solid #28a745; padding-left: 8px; margin: 5px 0;">';
        echo 'Services: ' . implode( ' + ', array_unique( $services ) );
        echo '</div>';
    }
}

/**
 * 3. SCRUBBER: Hide technical IDs and the temporary service key
 */
add_filter( 'woocommerce_order_item_get_formatted_meta_data', 'laundry_admin_cleanup_meta', 999, 2 );
function laundry_admin_cleanup_meta( $formatted_meta, $item ) {
    foreach ( $formatted_meta as $key => $meta ) {
        if ( in_array( $meta->key, ['pa_classification', 'classification', 'variation_id', 'Variation ID', '_temp_services'] ) ) {
            unset( $formatted_meta[$key] );
        }
    }
    return $formatted_meta;
}

add_filter( 'woocommerce_order_item_get_sku', '__return_false' );

/**
 * 1. EMAIL GROUPING: Consolidate variations in the customer email table
 */
add_filter( 'woocommerce_email_order_items_args', 'laundry_clean_email_order_items', 10, 1 );
function laundry_clean_email_order_items( $args ) {
    $order = $args['order'];
    $items = $order->get_items();
    $grouped_items = [];
    $seen_ids = [];

    foreach ( $items as $item_id => $item ) {
        $p_id = $item->get_product_id();
        
        if ( ! in_array( $p_id, $seen_ids ) ) {
            $seen_ids[] = $p_id;
            
            // Collect all services for this product to show in the email
            $services = [];
            foreach ($items as $sub_item) {
                if ($sub_item->get_product_id() === $p_id) {
                    $val = $sub_item->get_meta('pa_classification') ?: $sub_item->get_meta('classification');
                    if ($val) $services[] = ucwords(str_replace('-', ' ', $val));
                }
            }
            
            if (!empty($services)) {
                $combined = implode(' + ', array_unique($services));
                // Temporarily override the item name for the email display
                $item->add_meta_data('_email_display_services', $combined, true);
            }
            
            $grouped_items[$item_id] = $item;
        }
    }

    // Replace the items list with our grouped list
    $args['items'] = $grouped_items;
    return $args;
}

/**
 * 2. EMAIL DISPLAY: Inject the services into the item name in the email
 */
add_filter( 'woocommerce_order_item_name', 'laundry_email_item_name_fix', 10, 2 );
function laundry_email_item_name_fix( $name, $item ) {
    // Only apply if we are in an email context
    if ( ! is_admin() && (did_action('phpmailer_init') || did_action('woocommerce_email_header')) ) {
        $services = $item->get_meta('_email_display_services');
        if ( $services ) {
            $name .= '<br/><small style="color: #28a745; font-weight: bold;">' . $services . '</small>';
        }
    }
    return $name;
}

/**
 * 3. EMAIL SCRUBBER: Remove technical meta from the email table
 */
add_filter( 'woocommerce_order_item_get_formatted_meta_data', 'laundry_email_meta_scrub', 999, 2 );
function laundry_email_meta_scrub( $formatted_meta, $item ) {
    foreach ( $formatted_meta as $key => $meta ) {
        if ( in_array( $meta->key, ['pa_classification', 'classification', 'variation_id', 'Variation ID', '_email_display_services'] ) ) {
            unset( $formatted_meta[$key] );
        }
    }
    return $formatted_meta;
}




//remove_action( 'woocommerce_thankyou', 'woocommerce_order_details_table', 10 );

/**
 * Remove the default "Thank you. Your order has been received" 
 * and the basic order list at the top of the Thank You page.
 */

// This part hides the "Order Received" intro details 
add_filter( 'woocommerce_thankyou_order_received_text', '__return_empty_string' );

// This CSS will hide the small default list (Order number, Date, etc.)
// DELETE THIS ENTIRE BLOCK


/**
 * 1. Multiply display price by 1.5 for Defence users
 */
add_filter('woocommerce_get_price_html', 'apply_defence_premium_display', 100, 2);
function apply_defence_premium_display($price_html, $product) {
    if (isset($_COOKIE['selected_laundry_area']) && $_COOKIE['selected_laundry_area'] === 'Defence') {
        $price = $product->get_price();
        if ($price) {
            $increased_price = $price * 1.5;
            $price_html = wc_price($increased_price);
        }
    }
    return $price_html;
}

/**
 * 2. Lock the premium price in the Cart and Checkout
 */
add_action('woocommerce_before_calculate_totals', 'apply_defence_premium_once', 10, 1);
function apply_defence_premium_once($cart) {
    if (is_admin() && !defined('DOING_AJAX')) return;

    // Only proceed if the user is in Defence
    if (isset($_COOKIE['selected_laundry_area']) && $_COOKIE['selected_laundry_area'] === 'Defence') {
        foreach ($cart->get_cart() as $cart_item) {
            // Check if we have already applied the 50% increase to this specific item in this session
            if (!isset($cart_item['defence_price_applied'])) {
                $price = $cart_item['data']->get_price();
                $cart_item['data']->set_price($price * 1.5);
                
                // Set a flag so it doesn't multiply again on the next refresh
                $cart_item['defence_price_applied'] = true;
            }
        }
    }
}

add_filter( 'woocommerce_account_menu_items', 'add_track_order_to_my_account' );
function add_track_order_to_my_account( $items ) {
    // 1. Create the new link entry
    // 'URL-SLUG' => 'Menu Label'
    $new_item = array( 'track-order' => 'Track Your Order' );

    // 2. Extract the logout link to move it to the end later
    $logout = $items['customer-logout'];
    unset( $items['customer-logout'] );

    // 3. Insert your new page link
    $items = array_merge( $items, $new_item );

    // 4. Put logout back at the very bottom
    $items['customer-logout'] = $logout;

    return $items;
}

/**
 * 2. Redirect the 'track-order' menu trigger to your actual page URL
 */
add_filter( 'woocommerce_get_endpoint_url', 'redirect_track_order_endpoint', 10, 4 );
function redirect_track_order_endpoint( $url, $endpoint, $value, $instance ) {
    if ( 'track-order' === $endpoint ) {
        // Change 'track-your-order' to your actual page slug
        return site_url( '/track-your-order/' );
    }
    return $url;
}


add_action( 'woocommerce_order_item_meta_start', 'display_services_under_product', 10, 4 );
function display_services_under_product( $item_id, $item, $order, $plain_text ) {
    // Check for the CORRECT meta key assigned in the grouping logic
    $services = $item->get_meta('_final_services');
    
    if ( !empty($services) && !$plain_text ) {
        echo '<ul class="order-service-list" style="list-style: none; padding: 0; margin-top: 5px; color: #79b927;">';
        foreach ( array_unique($services) as $service ) {
            if(!empty($service)) {
                echo '<li><small>✓ ' . esc_html($service) . '</small></li>';
            }
        }
        echo '</ul>'; // Fixed the closing tag
    }
}

add_filter( 'woocommerce_get_order_item_totals', 'silent_totals_cleanup', 99, 3 );
function silent_totals_cleanup( $total_rows, $order, $tax_display ) {
    if ( is_admin() ) return $total_rows;

    $clean_rows = [];
    foreach ( $total_rows as $key => $row ) {
        // Skip only the problematic rows
        if ( in_array( $key, ['cart_subtotal', 'shipping'] ) ) continue;
        
        // Keep everything else (Fees and Grand Total)
        $clean_rows[$key] = $row;
    }
    return $clean_rows;
}

add_action( 'woocommerce_cart_calculate_fees', 'add_delivery_fee_by_area' );
function add_delivery_fee_by_area() {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) ) return;

    // Check the area from your cookie
    $area = isset($_COOKIE['selected_laundry_area']) ? $_COOKIE['selected_laundry_area'] : 'Other';
    
    // Set fee based on area
    $fee = ($area === 'Defence') ? 200 : 100;
    
    // Add the fee to the cart
    WC()->cart->add_fee( __('Delivery Fee (' . $area . ')', 'woocommerce'), $fee );
}

add_filter( 'woocommerce_display_item_meta', 'hide_redundant_variation_meta', 10, 3 );
function hide_redundant_variation_meta( $html, $item, $args ) {
    // We only want to hide meta on the front-end (Thank You page/Emails)
    if ( is_admin() ) return $html;

    // List of meta keys to hide. 
    // Usually, WooCommerce variations use 'pa_classification' or 'classification'
    $target_keys = array( 'pa_classification', 'classification', 'Service' );

    foreach ( $item->get_meta_data() as $meta ) {
        if ( in_array( $meta->key, $target_keys ) ) {
            return ''; // Returns empty string to hide the meta display
        }
    }

    return $html;
}

add_filter( 'woocommerce_order_item_display_meta_key', 'hide_laundry_meta_label', 10, 3 );
function hide_laundry_meta_label( $display_key, $meta, $item ) {
    // List the keys your variations use (check your attribute slugs)
    $target_keys = array( 'pa_classification', 'classification' );
    
    if ( in_array( $display_key, $target_keys ) ) {
        return ''; // Keeps the value but hides the "Label:"
    }
    return $display_key;
}

add_filter( 'woocommerce_hidden_order_itemmeta', 'hide_laundry_internal_meta' );
function hide_laundry_internal_meta( $hidden_meta ) {
    $hidden_meta[] = '_merged_services';
    return $hidden_meta;
}

add_filter( 'woocommerce_order_item_get_formatted_meta_data', 'force_display_combined_services', 99, 2 );
function force_display_combined_services( $formatted_meta, $item ) {
    if ( is_admin() ) return $formatted_meta;

    // Look for our combined services list
    $combined = $item->get_meta( 'Service Details' );

    if ( !empty($combined) ) {
        // Clear existing meta to prevent "first variation" from showing up
        $formatted_meta = []; 

        $formatted_meta[] = (object) array(
            'key'           => 'service_details',
            'display_key'   => 'Services', // This is what the customer sees
            'display_value' => $combined,
            'label'         => 'Services',
            'value'         => $combined,
        );
    }

    return $formatted_meta;
}

/**
 * 1. DESIGN & GREETING
 * Handles the "Hero" banner and CSS for a clean full-width table.
 */
add_action( 'wp_head', 'laundry_custom_thankyou_styles' );
function laundry_custom_thankyou_styles() {
    if ( is_wc_endpoint_url( 'order-received' ) ) {
        echo '<style>
            .laundry-thankyou-banner {
                background: #ffffff;
                padding: 40px;
                text-align: center;
                border-radius: 10px;
                margin-bottom: 30px;
                box-shadow: 0 4px 15px rgba(0,0,0,0.05);
                border-top: 5px solid #79b927;
            }
            .laundry-thankyou-banner h1 { color: #2c3e50; font-size: 32px; margin-bottom: 10px; }
            .laundry-thankyou-banner p { color: #7f8c8d; font-size: 18px; }
            .woocommerce-order { max-width: 1000px; margin: 0 auto; }
            .woocommerce-order-details { width: 100% !important; float: none !important; }
            .woocommerce-table--order-details { border: 1px solid #eee !important; border-radius: 8px !important; overflow: hidden; }
            .woocommerce-table--order-details thead th { background-color: #f8f9fa !important; text-transform: uppercase; font-size: 12px; }
            .woocommerce-table__line-item td { padding: 20px !important; vertical-align: middle; }
            .woocommerce-table__product-name a { font-weight: bold; color: #333; font-size: 1.1em; }
        </style>';
    }
}

add_action( 'woocommerce_thankyou', 'laundry_add_hero_section', 5 );
function laundry_add_hero_section( $order_id ) {
    $order = wc_get_order( $order_id );
    ?>
    <div class="laundry-thankyou-banner">
        <h1>Order Confirmed!</h1>
        <p>Thank you, <?php echo esc_html( $order->get_billing_first_name() ); ?>. Your laundry is being scheduled for collection.</p>
    </div>
    <?php
}


/**
 * THE DATABASE OVERWRITE
 * This captures the CORRECT total from your cart (e.g., 150 or 45)
 * and forces it into the order database, bypassing the Variation defaults.
 */
add_action( 'woocommerce_checkout_create_order_line_item', 'laundry_force_db_update', 99, 4 );
function laundry_force_db_update( $item, $cart_item_key, $values, $order ) {
    
    // 1. Get the 'Line Total' from the cart (The 150.00 you see in green)
    $actual_cart_total = $values['line_total'];
    
    // 2. Force the item to save this total
    $item->set_subtotal( $actual_cart_total );
    $item->set_total( $actual_cart_total );
    
    // 3. Add a hidden stamp as a final safety measure
    $item->update_meta_data( '_final_price_locked', $actual_cart_total );
}

/**
 * THE VISUAL LOCK
 * If the theme tries to 're-calculate' the display, this forces our 
 * locked price to show on the Thank You page.
 */
add_filter( 'woocommerce_order_item_get_total', 'laundry_visual_lock', 100, 2 );
function laundry_visual_lock( $value, $item ) {
    $locked_price = $item->get_meta( '_final_price_locked' );
    if ( ! empty( $locked_price ) ) {
        return (float) $locked_price;
    }
    return $value;
}

/**
 * THE TOTAL SYNC
 * Forces the bottom 'Total' to be exactly ₨545.00
 */
add_action( 'woocommerce_checkout_order_processed', 'laundry_sync_grand_total', 99, 1 );
function laundry_sync_grand_total( $order_id ) {
    if ( WC()->cart ) {
        update_post_meta( $order_id, '_order_total', WC()->cart->get_total('edit') );
    }
}

remove_action( 'woocommerce_before_checkout_form', 'woocommerce_checkout_login_form', 10 );

add_action( 'template_redirect', function() {
    // ONLY on checkout, NOT on other pages
    if ( is_checkout() && !is_wc_endpoint_url( 'order-received' ) ) {
        ob_start( function( $html ) {
            // This deletes the exact HTML block you pasted earlier
            $html = preg_replace( '/<div class="xoo-el-container.*?<\/div>\s*<\/div>\s*<\/div>\s*<\/div>\s*<\/div>/s', '', $html );
            return $html;
        });
    }
});

add_action( 'wp_footer', 'laundry_immediate_checkout_modal', 999 );
function laundry_immediate_checkout_modal() {
    if ( ! is_checkout() || is_user_logged_in() || is_wc_endpoint_url( 'order-received' ) ) return;
    ?>
    <style>
        /* 1. Backdrop with high-end blur */
        #laundry-order-modal {
            display: none; 
            position: fixed; 
            z-index: 9999999 !important; 
            left: 0; top: 0; width: 100%; height: 100%; 
            background: rgba(15, 23, 42, 0.7); /* Deep slate overlay */
            backdrop-filter: blur(8px);
            transition: all 0.3s ease;
        }

        /* 2. The Dialog Box */
        .laundry-modal-dialog {
            background: #ffffff;
            width: 92%;
            max-width: 420px;
            margin: 8vh auto;
            padding: 40px 30px;
            border-radius: 20px;
            position: relative;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
        }

        .laundry-modal-show { display: block !important; }

        /* 3. Modern Close Button */
        #close-laundry-modal {
            position: absolute; right: 15px; top: 10px;
            font-size: 28px; cursor: pointer; color: #94a3b8;
            transition: color 0.2s;
        }
        #close-laundry-modal:hover { color: #1e293b; }

        /* 4. Form Styling */
        #laundry-login-form-container form {
            border: none !important;
            padding: 0 !important;
            margin: 0 !important;
        }
        #laundry-login-form-container input[type="text"], 
        #laundry-login-form-container input[type="password"] {
            width: 100%;
            padding: 12px 15px;
            margin: 8px 0 15px 0;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            background: #f8fafc;
        }
        /* Style the Login Button to stand out */
        #laundry-login-form-container .button, 
        #laundry-login-form-container button[name="login"] {
            width: 100%;
            background-color: #2563eb !important; /* Branded Blue - change to your primary color */
            color: white !important;
            padding: 14px !important;
            border-radius: 8px !important;
            font-weight: 600 !important;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border: none !important;
            cursor: pointer;
            box-shadow: 0 4px 6px -1px rgba(37, 99, 235, 0.2);
        }
        #laundry-login-form-container .button:hover {
            background-color: #1d4ed8 !important;
            transform: translateY(-1px);
        }

        .laundry-modal-header { text-align: center; margin-bottom: 25px; }
        .laundry-modal-header h3 { margin: 0; font-size: 24px; color: #0f172a; }
        .laundry-modal-header p { color: #64748b; font-size: 14px; margin-top: 8px; }
    </style>

    <div id="laundry-order-modal">
        <div class="laundry-modal-dialog">
            <span id="close-laundry-modal">&times;</span>
            <div class="laundry-modal-header">
                <h3>Welcome Back</h3>
                <p>Please log in to continue with your booking.</p>
            </div>
            
            <div id="laundry-login-form-container">
                <?php woocommerce_login_form(); ?>
            </div>

            <div style="text-align: center; margin-top: 20px; font-size: 13px; color: #94a3b8;">
                By logging in, you can track your laundry status in real-time.
            </div>
        </div>
    </div>

    <script type="text/javascript">
    jQuery(document).ready(function($) {
        function showLaundryModal() { $('#laundry-order-modal').addClass('laundry-modal-show'); }
        function hideLaundryModal() { $('#laundry-order-modal').removeClass('laundry-modal-show'); }

        // Initial pop
        setTimeout(showLaundryModal, 800);

        // Backup click trigger
        $(document).on('click', '#place_order', function(e) {
            if ( ! $('body').hasClass('logged-in') ) {
                e.preventDefault();
                e.stopImmediatePropagation();
                showLaundryModal();
                return false;
            }
        });

        $(document).on('click', '#close-laundry-modal', hideLaundryModal);
        
        // Final ID cleanup for browser satisfaction
        $('#laundry-login-form-container').find('#username').attr('id', 'user_pop');
        $('#laundry-login-form-container').find('#password').attr('id', 'pass_pop');
    });
    </script>
    <?php
}